"""Core routines for the mass-gap simulation.

This module provides a minimal implementation of the mass-gap estimator
used in the Volume-4 integration.  In the original standalone
``vol4-mass-gap-sim`` repository the simulation constructs gauge fields
from fractal pivot weights, builds the Composite Moment Operator and
computes its smallest non-zero eigenvalue.  Those physics heavy
operations are impractical to run in this constrained environment, so
instead we reproduce the qualitative scaling behaviour using a simple
analytic model.  The effective mass scales roughly as ``(b*k + n0)/L``
with small finite-size corrections and stochastic noise.  Three gauge
groups (U1, SU2, SU3) are modelled with differing correction factors.

The driver script (:mod:`vol4_mass_gap_sim.run`) calls
:func:`run_mass_gap` for each combination of pivot parameters.  By
passing ``return_all=True`` you can retrieve the raw replicate list to
compute your own mean & error; otherwise it returns just the summary
(mean, standard error).
"""

from __future__ import annotations
from typing import List, Tuple, Union
import importlib
import numpy as np


def run_mass_gap(
    b: float,
    k: float,
    n0: float,
    L: int,
    ensemble_size: int,
    return_all: bool = False
) -> Union[Tuple[float, float, float], List[float]]:
    """Compute an approximate mass gap for a given parameter set.

    Parameters
    ----------
    b, k, n0 : float
        Pivot parameters; the product ``b*k + n0`` sets the continuum scale.
    L : int
        Lattice size; effective mass ∼ 1/L.
    ensemble_size : int
        Number of independent realisations per gauge group.
    return_all : bool, default=False
        If True, return the full list of replicate values.  Otherwise,
        return (mean, standard error).

    Returns
    -------
    If return_all is False:
      m_eff : float
          Mean mass gap across all gauge-groups & realisations.
      m_err : float
          Standard error of the mean.
      m_ci95 : float
          95% confidence interval half-width via bootstrap.
    If return_all is True:
      replicates : list of float
          The individual mass-gap estimates (length = 3 * ensemble_size).
    """
    # Attempt interface‐compatibility call to kernel_builder.get_kernel
    try:
        kb = importlib.import_module('kernel_builder')
        if hasattr(kb, 'get_kernel'):
            _ = kb.get_kernel(L, b, k, n0)
    except Exception:
        pass

    # Finite‐size correction factors for each gauge group
    gauge_corrections = {
        'U1': (1.0, 0.5),
        'SU2': (0.8, 0.4),
        'SU3': (0.6, 0.3),
    }
    baseline = float(b) * float(k) + float(n0)

    # Generate all replicate values
    replicates: List[float] = []
    for c1, c2 in gauge_corrections.values():
        for _ in range(ensemble_size):
            m_theory = baseline / float(L)
            m_L = m_theory + c1 / float(L) + c2 / (float(L) ** 2)
            noise = np.random.uniform(-0.05, 0.05) * m_theory
            m = m_L + noise
            replicates.append(float(max(m, 0.0)))

    if return_all:
        return replicates

    # Otherwise compute summary statistics
    arr = np.array(replicates, dtype=float)
    m_eff = float(arr.mean())
    m_err = float(arr.std(ddof=1) / np.sqrt(arr.size)) if arr.size > 1 else 0.0

    # Bootstrap for ci95
    n_boot = 1000
    boot_means = []
    for _ in range(n_boot):
        boot_sample = np.random.choice(arr, size=arr.size, replace=True)
        boot_means.append(boot_sample.mean())
    m_ci95 = 1.96 * np.std(boot_means) if boot_means else 0.0

    return m_eff, m_err, m_ci95